const { spawn, execSync } = require("child_process");
const { WebSocketServer } = require("ws");
const logger = require("./logger");

module.exports = class MicrophoneServer {
  constructor({ server }) {
    const path = `/microphone`;
    //logger.info(`Microphone websocket server starting`, path);  // 减少启动日志
    const wss = new WebSocketServer(
      {
        noServer: true,
        path,
      },
      () => {
        //logger.info(`Microphone websocket server started`, path);  // 减少启动日志
      }
    );

    server.on("upgrade", (request, socket, head) => {
      if (request.url === path)
        wss.handleUpgrade(request, socket, head, (ws) => {
          wss.emit("connection", ws, request);
        });
    });

    this.wss = wss;
    this.streamer = undefined;
    this.streamConfig = undefined;
    this.retryTimer = undefined;
    this.retryDelay = Number(process.env.NRC_MIC_RETRY_MS || 1000);
    this.opening = false;
    this.detectedDevice = null;
    this.detectedAt = 0;
    this.detectedChannels = null;
    this.detectedChannelsAt = 0;
    this.detectTtl = Number(process.env.NRC_MIC_DETECT_TTL_MS || 10000);
    this.deviceCandidates = [];
    this.deviceIndex = 0;
    this.lastDataAt = 0;
    this.noDataTimer = undefined;
    this.noDataTimeout = Number(process.env.NRC_MIC_NO_DATA_MS || 3000);
    this.totalBytes = 0;
    this.statsTimer = undefined;

    wss.on("connection", (socket) => {
      //logger.info(`客户端已连接  Microphone Websocket Server`);  // 减少连接日志
      if (this.streamConfig) {
        socket.send(
          JSON.stringify({ action: "audio-config", payload: this.streamConfig })
        );
      }
      socket.on("close", () => {
        if (wss.clients.size === 0) {
          this.close();
        }
      });

      // 如果有客户端连接，则启动麦克风流
      this.open();
    });

    wss.on("error", () => {
      logger.error(`麦克风服务错误`);
    });
  }

  clearRetry() {
    if (this.retryTimer) {
      clearTimeout(this.retryTimer);
      this.retryTimer = undefined;
    }
  }

  clearNoDataTimer() {
    if (this.noDataTimer) {
      clearInterval(this.noDataTimer);
      this.noDataTimer = undefined;
    }
  }

  clearStatsTimer() {
    if (this.statsTimer) {
      clearInterval(this.statsTimer);
      this.statsTimer = undefined;
    }
  }

  startNoDataTimer() {
    this.clearNoDataTimer();
    if (!this.noDataTimeout) return;
    this.noDataTimer = setInterval(() => {
      if (!this.streamer) return;
      if (!this.lastDataAt) {
        this.rotateDeviceAndRestart("无音频数据");
        return;
      }
      const gap = Date.now() - this.lastDataAt;
      if (gap > this.noDataTimeout) {
        this.rotateDeviceAndRestart(`音频中断${gap}ms`);
      }
    }, this.noDataTimeout).unref?.();
  }

  rotateDeviceAndRestart(reason) {
    if (!this.deviceCandidates.length) return;
    if (this.deviceCandidates.length === 1) return;
    this.deviceIndex = (this.deviceIndex + 1) % this.deviceCandidates.length;
    const nextDevice = this.deviceCandidates[this.deviceIndex];
    logger.warn(`麦克风无数据，切换设备: ${nextDevice}${reason ? `，原因：${reason}` : ""}`);
    this.close();
    setTimeout(() => this.open(nextDevice), 200).unref?.();
  }

  resolveDevice() {
    const envDevice = process.env.NRC_MIC_DEVICE;
    if (envDevice) {
      if (envDevice.startsWith("hw:")) {
        return `plughw:${envDevice.slice(3)}`;
      }
      return envDevice;
    }

    const now = Date.now();
    if (this.detectedDevice && now - this.detectedAt < this.detectTtl) {
      return this.detectedDevice;
    }

    let device = "default";
    try {
      const output = execSync("arecord -l 2>/dev/null", {
        timeout: 1500,
        stdio: ["ignore", "pipe", "pipe"],
        maxBuffer: 1024 * 1024,
      }).toString();
      const lines = output.split("\n").filter((line) => line.includes("card "));
      const candidates = [];

      lines.forEach((line) => {
        const match = line.match(
          /card\s+(\d+):\s*([^\[]+)\[([^\]]+)\],\s*device\s+(\d+):\s*([^\[]+)\[/
        );
        if (!match) return;
        const [, cardId, cardName, cardDesc, devId, devName] = match;
        const label = `${cardName} ${cardDesc} ${devName}`.toLowerCase();
        const isUsb = label.includes("usb");
        const isMic = label.includes("mic") || label.includes("microphone");
        const isAudio = label.includes("audio");
        const score = (isUsb ? 3 : 0) + (isMic ? 2 : 0) + (isAudio ? 1 : 0);
        candidates.push({
          device: `plughw:${cardId},${devId}`,
          score,
          label,
        });
      });

      candidates.sort((a, b) => b.score - a.score);
      this.deviceCandidates = candidates.map((c) => c.device);
      if (candidates.length) {
        if (this.deviceIndex >= candidates.length) this.deviceIndex = 0;
        device = candidates[this.deviceIndex].device;
      }
    } catch (e) {
      // ignore
    }

    if (!this.deviceCandidates.length) {
      this.deviceCandidates = ["default"];
      if (this.deviceIndex >= this.deviceCandidates.length) this.deviceIndex = 0;
      device = this.deviceCandidates[this.deviceIndex];
    }

    this.detectedDevice = device;
    this.detectedAt = now;
    logger.info(`自动选择麦克风设备: ${device}`);
    return device;
  }

  resolveChannels(device, sampleRate) {
    const envChannels = process.env.NRC_MIC_CHANNELS;
    if (envChannels) return Number(envChannels);

    const now = Date.now();
    if (this.detectedChannels && now - this.detectedChannelsAt < this.detectTtl) {
      return this.detectedChannels;
    }

    let channels = 1;
    const timeoutMs = Number(process.env.NRC_MIC_DETECT_TIMEOUT_MS || 1500);
    try {
      const output = execSync(
        `arecord -D ${device} --dump-hw-params -f S16_LE -r ${sampleRate} 2>/dev/null`,
        {
          timeout: timeoutMs,
          stdio: ["ignore", "pipe", "pipe"],
          maxBuffer: 1024 * 1024,
        }
      ).toString();
      const line = output
        .split("\n")
        .find((l) => l.toLowerCase().includes("channels"));
      if (line) {
        const nums = (line.match(/\d+/g) || []).map((n) => Number(n));
        if (nums.length) {
          channels = nums.includes(1) ? 1 : nums[0];
        }
      }
    } catch (e) {
      if (process.env.NRC_MIC_LOG === "1") {
        logger.warn("麦克风通道检测超时或失败，使用默认通道数");
      }
    }

    this.detectedChannels = channels;
    this.detectedChannelsAt = now;
    logger.info(`自动选择麦克风通道数: ${channels}`);
    return channels;
  }

  scheduleRetry(reason) {
    if (this.retryTimer || this.wss.clients.size === 0) return;
    this.retryTimer = setTimeout(() => {
      this.retryTimer = undefined;
      this.open();
    }, this.retryDelay);
    logger.warn(
      `麦克风重试(${this.retryDelay}ms)${reason ? `，原因：${reason}` : ""}`
    );
  }

  // 供外部调用的方法来启动/停止麦克风
  handleAction(action) {
    if (action === 'start mic') {
      this.open();
    } else if (action === 'stop mic') {
      this.close();
    }
  }

  forceClaimDevice() {
    const enabled = String(process.env.NRC_MIC_FORCE_CLAIM ?? "1").toLowerCase();
    if (!"1,true,yes,on".split(",").includes(enabled)) return;
    try {
      execSync("fuser -k /dev/snd/pcmC*D*c 2>/dev/null || true");
      execSync("pkill -f 'ffmpeg -f alsa' 2>/dev/null || true");
      execSync("pkill -f 'arecord' 2>/dev/null || true");
      logger.info("已尝试抢占麦克风设备");
    } catch (e) {
      // ignore
    }
  }

  async open(deviceOverride) {
    //logger.info(`Microphone start`);  // 减少启动日志
    if (this.streamer || this.opening) return;
    this.opening = true;
    this.clearRetry();
    this.clearNoDataTimer();
    this.clearStatsTimer();
    this.forceClaimDevice();

    const device = deviceOverride || this.resolveDevice();
    if (deviceOverride) {
      this.detectedDevice = deviceOverride;
      this.detectedAt = Date.now();
    }
    const sampleRate = Number(process.env.NRC_MIC_SAMPLE_RATE || 16000);
    const channels = this.resolveChannels(device, sampleRate);
    this.streamConfig = { sampleRate, channels, format: "s16le" };
    this.broadcast("audio-config", this.streamConfig);
    const { proc: streamer, label: captureLabel } = createCapture(
      device,
      sampleRate,
      channels
    );
    this.streamer = streamer;
    this.captureLabel = captureLabel;
    this.lastDataAt = 0;
    this.totalBytes = 0;
    this.startNoDataTimer();

    if (process.env.NRC_MIC_LOG === "1") {
      this.statsTimer = setInterval(() => {
        logger.info(`麦克风流量: ${this.totalBytes} bytes`);
        this.totalBytes = 0;
      }, 5000).unref?.();
    }

    streamer.stdout.on("data", (data) => {
      if (!this.lastDataAt) {
        logger.info("麦克风音频流已开始输出");
      }
      this.lastDataAt = Date.now();
      this.totalBytes += data.length;
      this.broadcastStream(data);
    });

    const handleExit = (reason) => {
      if (this.streamer === streamer) {
        this.streamer = undefined;
      }
      this.opening = false;
      this.clearNoDataTimer();
      this.clearStatsTimer();
      if (this.wss.clients.size > 0) {
        this.scheduleRetry(reason);
      }
    };

    streamer.on("close", (code) => {
      handleExit(`${captureLabel} 退出(${code})`);
    });

    streamer.on("error", (err) => {
      handleExit(`${captureLabel} 错误: ${err.message}`);
    });
  }

  close() {
    this.clearRetry();
    const streamer = this.streamer;
    this.streamer = undefined;
    this.opening = false;
    if (streamer) {
      const signal = this.captureLabel === "arecord" ? "SIGINT" : "SIGHUP";
      streamer.kill(signal);
      setTimeout(() => {
        if (!streamer.killed) {
          streamer.kill("SIGKILL");
        }
      }, 1500).unref?.();
    }
  }

  sendBinary(socket, frame) {
    if (socket.buzy) return;
    socket.buzy = true;
    socket.buzy = false;

    socket.send(frame, { binary: true }, function ack() {
      socket.buzy = false;
    });
  }

  broadcast(action, payload) {
    this.wss.clients.forEach((socket) =>
      socket.send(JSON.stringify({ action, payload }))
    );
  }

  broadcastStream(data) {
    this.wss.clients.forEach((socket) => {
      this.sendBinary(socket, data);
    });
  }
};

const createCapture = function (deviceOverride, sampleRateOverride, channelsOverride) {
  const device = deviceOverride || process.env.NRC_MIC_DEVICE || "default";
  const sampleRate = Number(sampleRateOverride || process.env.NRC_MIC_SAMPLE_RATE || 16000);
  const channels = Number(channelsOverride || process.env.NRC_MIC_CHANNELS || 1);
  const capture = String(process.env.NRC_MIC_CAPTURE || "arecord").toLowerCase();

  if (capture === "ffmpeg") {
    const proc = spawn("ffmpeg", [
      "-f",
      "alsa",
      "-thread_queue_size",
      "4096",
      "-ar",
      sampleRate,
      "-ac",
      channels,
      "-i",
      device,
      "-acodec",
      "pcm_s16le",
      "-f",
      "s16le",
      "-",
    ]);

    proc.on("error", (err) => {
      logger.error(`麦克风服务错误: ${err.message}`);
    });

    proc.stderr.on("data", (data) => {
      const msg = data.toString();
      if (
        msg.toLowerCase().includes("error") ||
        msg.toLowerCase().includes("cannot") ||
        msg.toLowerCase().includes("failed")
      ) {
        logger.error(`麦克风服务错误: ${msg}`);
      } else if (process.env.NRC_MIC_LOG === "1") {
        //logger.info(`Microphone ffmpeg: ${msg}`);  // 减少日志
      }
    });

    return { proc, label: "ffmpeg" };
  }

  const args = [
    "-D",
    device,
    "-f",
    "S16_LE",
    "-r",
    String(sampleRate),
    "-c",
    String(channels),
    "-t",
    "raw",
  ];
  const proc = spawn("arecord", args);
  proc.on("error", (err) => {
    logger.error(`麦克风服务错误: ${err.message}`);
  });
  proc.stderr.on("data", (data) => {
    const msg = data.toString();
    if (
      msg.toLowerCase().includes("error") ||
      msg.toLowerCase().includes("cannot") ||
      msg.toLowerCase().includes("failed") ||
      msg.toLowerCase().includes("unable")
    ) {
      logger.error(`麦克风服务错误: ${msg}`);
    } else if (process.env.NRC_MIC_LOG === "1") {
      //logger.info(`Microphone arecord: ${msg}`);  // 减少日志
    }
  });
  return { proc, label: "arecord" };
};