import React, { useState, useEffect } from "react";
import { Form, InputNumber, Switch, Select, Card, Space, Input } from "antd";
import { CloseOutlined, DragOutlined } from "@ant-design/icons";

const { Option } = Select;

export default function DraggableEncoderComponent({ 
  component, 
  onUpdate, 
  onDelete,
  isDragging,
  onDragStart 
}) {
  const [form] = Form.useForm();
  
  useEffect(() => {
    if (component) {
      form.setFieldsValue({
        name: component.name,
        enabled: component.enabled,
        ...component.config
      });
    }
  }, [component, form]);

  const handleValuesChange = (changedValues) => {
    const updatedComponent = { ...component, ...changedValues };
    if (changedValues.config) {
      updatedComponent.config = { ...component.config, ...changedValues.config };
    }
    onUpdate(updatedComponent);
  };

  return (
    <Card
      className="draggable-component"
      title={
        <Space>
          <DragOutlined 
            onMouseDown={onDragStart}
            style={{ cursor: 'move', opacity: isDragging ? 0.5 : 1 }}
          />
          <span>{component.name}</span>
        </Space>
      }
      extra={
        <CloseOutlined 
          onClick={() => onDelete(component.id)} 
          style={{ cursor: 'pointer' }} 
        />
      }
      size="small"
    >
      <Form
        form={form}
        layout="vertical"
        onValuesChange={handleValuesChange}
        initialValues={{
          name: component.name,
          enabled: component.enabled,
          ...component.config
        }}
      >
        <Form.Item name="enabled" valuePropName="checked">
          <Switch checkedChildren="启用" unCheckedChildren="禁用" />
        </Form.Item>

        <Form.Item name="name" label="组件名称">
          <Input placeholder="编码器显示" />
        </Form.Item>

        <Form.Item label="编码器配置">
          <Form.Item name={['config', 'gpioAPin']} label="GPIO A 引脚" initialValue={5}>
            <InputNumber min={0} max={40} disabled={true} /> {/* 固定值，不可修改 */}
          </Form.Item>

          <Form.Item name={['config', 'gpioBPin']} label="GPIO B 引脚" initialValue={6}>
            <InputNumber min={0} max={40} disabled={true} /> {/* 固定值，不可修改 */}
          </Form.Item>

          <Form.Item name={['config', 'ppr']} label="每转脉冲数 (PPR)">
            <InputNumber min={1} max={10000} />
          </Form.Item>

          <Form.Item name={['config', 'mode']} label="编码器模式">
            <Select>
              <Option value="x1">x1</Option>
              <Option value="x2">x2</Option>
              <Option value="x4">x4</Option>
            </Select>
          </Form.Item>

          <Form.Item name={['config', 'intervalMs']} label="更新间隔 (毫秒)">
            <InputNumber min={100} max={10000} />
          </Form.Item>

          <Form.Item name={['config', 'wheelDiameterMm']} label="轮子直径 (毫米)">
            <InputNumber min={10} max={1000} />
          </Form.Item>

          <Form.Item name={['config', 'speedUnit']} label="速度单位">
            <Select>
              <Option value="km/h">公里/小时 (km/h)</Option>
              <Option value="m/s">米/秒 (m/s)</Option>
              <Option value="mph">英里/小时 (mph)</Option>
            </Select>
          </Form.Item>

          <Form.Item name={['config', 'speedLabel']} label="车速显示标签">
            <Input placeholder="车速" />
          </Form.Item>

          <Form.Item name={['config', 'rpmLabel']} label="转速显示标签">
            <Input placeholder="转速" />
          </Form.Item>

          <Form.Item name={['config', 'showRpm']} label="显示转速" valuePropName="checked">
            <Switch />
          </Form.Item>

          <Form.Item name={['config', 'showDirection']} label="显示方向" valuePropName="checked">
            <Switch />
          </Form.Item>
        </Form.Item>
      </Form>
    </Card>
  );
}