import React, { useState, useEffect } from "react";
import { Card, Statistic, Row, Col, Divider } from "antd";
import { CarOutlined, RetweetOutlined, SwapOutlined } from "@ant-design/icons";

export default function SpeedDisplay({ ws, connected }) {
  const [speedData, setSpeedData] = useState({
    speed: "0.00",
    unit: "km/h",
    rpm: "0.00",
    direction: "停止",
    timestamp: Date.now()
  });

  useEffect(() => {
    if (!ws || !connected) return;

    const handleSpeedUpdate = (data) => {
      setSpeedData(prev => ({
        ...prev,
        ...data,
        timestamp: Date.now()
      }));
    };

    // 监听速度数据更新
    const handleMessage = (event) => {
      try {
        const message = JSON.parse(event.data);
        if (message.action === 'encoder-data' || message.type === 'encoder-data') {
          handleSpeedUpdate(message.payload || message.data);
        }
      } catch (e) {
        // 忽略非JSON消息
      }
    };

    ws.addEventListener('message', handleMessage);

    return () => {
      ws.removeEventListener('message', handleMessage);
    };
  }, [ws, connected]);

  // 计算最后一次更新的时间
  const timeDiff = Date.now() - speedData.timestamp;
  const isStale = timeDiff > 3000; // 3秒后认为数据过时

  return (
    <Card 
      title="车速信息" 
      size="small"
      style={{ 
        width: '100%', 
        background: isStale ? '#f0f0f0' : '#ffffff',
        opacity: isStale ? 0.7 : 1
      }}
    >
      <Row gutter={[16, 16]}>
        <Col span={12}>
          <Statistic
            title={
              <span style={{ fontSize: '14px', fontWeight: 'bold' }}>
                <CarOutlined style={{ marginRight: '5px', color: '#1890ff' }} />
                当前速度
              </span>
            }
            value={speedData.speed}
            precision={2}
            valueStyle={{ 
              color: '#3f8600', 
              fontSize: '20px',
              fontWeight: 'bold'
            }}
            prefix=""
            suffix={speedData.unit}
          />
        </Col>
        <Col span={12}>
          <Statistic
            title={
              <span style={{ fontSize: '14px', fontWeight: 'bold' }}>
                <RetweetOutlined style={{ marginRight: '5px', color: '#52c41a' }} />
                转速
              </span>
            }
            value={speedData.rpm}
            precision={2}
            valueStyle={{ 
              color: '#1890ff', 
              fontSize: '20px',
              fontWeight: 'bold'
            }}
            prefix=""
            suffix="RPM"
          />
        </Col>
      </Row>
      
      <Divider style={{ margin: '16px 0' }} />
      
      <Row justify="center">
        <Col>
          <div style={{
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            padding: '8px 16px',
            background: '#f6ffed',
            border: '1px solid #b7eb8f',
            borderRadius: '4px'
          }}>
            <SwapOutlined style={{ marginRight: '8px', color: '#52c41a' }} />
            <span style={{ fontWeight: 'bold', color: '#52c41a' }}>
              {speedData.direction}
            </span>
          </div>
        </Col>
      </Row>
      
      <div style={{
        marginTop: '12px',
        fontSize: '12px',
        color: '#888',
        textAlign: 'center'
      }}>
        {isStale ? '数据已停止更新' : '实时更新'}
      </div>
    </Card>
  );
}