const { spawn } = require("child_process");
const { WebSocketServer } = require("ws");
const logger = require("./logger");

module.exports = class MicrophoneServer {
  constructor({ server }) {
    const path = `/microphone`;
    //logger.info(`Microphone websocket server starting`, path);  // 减少启动日志
    const wss = new WebSocketServer(
      {
        noServer: true,
        path,
      },
      () => {
        //logger.info(`Microphone websocket server started`, path);  // 减少启动日志
      }
    );

    server.on("upgrade", (request, socket, head) => {
      if (request.url === path)
        wss.handleUpgrade(request, socket, head, (ws) => {
          wss.emit("connection", ws, request);
        });
    });

    this.wss = wss;
    this.streamer = undefined;
    this.streamConfig = undefined;

    wss.on("connection", (socket) => {
      //logger.info(`客户端已连接  Microphone Websocket Server`);  // 减少连接日志
      if (this.streamConfig) {
        socket.send(
          JSON.stringify({ action: "audio-config", payload: this.streamConfig })
        );
      }
      socket.on("close", () => {
        if (wss.clients.size === 0) {
          this.close();
        }
      });

      // 如果有客户端连接，则启动麦克风流
      this.open();
    });

    wss.on("error", (err) => {
      logger.error(`麦克风服务错误`);
    });
  }

  // 供外部调用的方法来启动/停止麦克风
  handleAction(action, payload) {
    if (action === 'start mic') {
      this.open();
    } else if (action === 'stop mic') {
      this.close();
    }
  }

  async open() {
    //logger.info(`Microphone start`);  // 减少启动日志
    if (this.streamer) {
      //logger.info(`Microphone ffmpeg streamer already open`);  // 减少日志
      this.close();
    }
    const { proc, config } = ffmpeg();
    this.streamer = proc;
    this.streamConfig = config;
    this.broadcast("audio-config", this.streamConfig);
    this.streamer.stdout.on("data", (data) => {
      this.broadcastStream(data);
    });
  }

  close() {
    if (this.streamer) {
      //logger.info(`Microphone ffmpeg streamer killing`);  // 减少日志
      this.streamer.kill("SIGHUP");
      this.streamer = undefined;
    }
  }

  sendBinary(socket, frame) {
    if (socket.buzy) return;
    socket.buzy = true;
    socket.buzy = false;

    socket.send(frame, { binary: true }, function ack() {
      socket.buzy = false;
    });
  }

  broadcast(action, payload) {
    this.wss.clients.forEach((socket) =>
      socket.send(JSON.stringify({ action, payload }))
    );
  }

  broadcastStream(data) {
    this.wss.clients.forEach((socket) => {
      this.sendBinary(socket, data);
    });
  }
};

const ffmpeg = function () {
  // 自动检测 USB 麦克风设备
  const device = process.env.NRC_MIC_DEVICE || "default";
  const sampleRate = Number(process.env.NRC_MIC_SAMPLE_RATE || 16000);
  const channels = Number(process.env.NRC_MIC_CHANNELS || 1);

  //logger.info(
  //  `Microphone ffmpeg start: device=${device}, sampleRate=${sampleRate}, channels=${channels}`
  //);  // 减少启动日志

  const proc = spawn("ffmpeg", [
    "-f",
    "alsa",
    "-thread_queue_size",
    "4096",
    "-i",
    device,
    "-acodec",
    "pcm_s16le",
    "-ar",
    sampleRate,
    "-ac",
    channels,
    "-f",
    "s16le",
    "-",
  ]);

  proc.on("close", (code) => {
    //logger.info(`Microphone ffmpeg close with code: ${code}`);  // 减少日志
  });

  proc.on("error", (err) => {
    logger.error(`麦克风服务错误: ${err.message}`);
  });

  proc.stderr.on("data", (data) => {
    const msg = data.toString();
    // 记录所有错误信息
    if (
      msg.toLowerCase().includes("error") ||
      msg.toLowerCase().includes("cannot") ||
      msg.toLowerCase().includes("failed")
    ) {
      logger.error(`麦克风服务错误: ${msg}`);
    } else if (process.env.NRC_MIC_LOG === "1") {
      //logger.info(`Microphone ffmpeg: ${msg}`);  // 减少日志
    }
  });

  return {
    proc,
    config: {
      sampleRate,
      channels,
      format: "s16le",
    },
  };
};