#!/usr/bin/env python3
"""
简化版DonkeyCar管理脚本 - 仅保留USB摄像头和电调舵机控制
"""
import os
import logging
import donkeycar as dk
from donkeycar.parts.controller import LocalWebController
from donkeycar.parts.camera import Webcam
from donkeycar.parts import actuator, pins
from donkeycar.parts.actuator import PWMSteering, PWMThrottle, PulseController
from donkeycar.parts.pipe import Pipe

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO)

def drive_simple(cfg):
    """
    简化的驾驶函数 - 仅保留必要的部件
    """
    logger.info(f'PID: {os.getpid()}')
    
    V = dk.vehicle.Vehicle()

    if cfg.HAVE_CONSOLE_LOGGING:
        logger.setLevel(logging.getLevelName(cfg.LOGGING_LEVEL))
        ch = logging.StreamHandler()
        ch.setFormatter(logging.Formatter(cfg.LOGGING_FORMAT))
        logger.addHandler(ch)

    # 添加USB摄像头
    logger.info("cfg.CAMERA_TYPE %s" % cfg.CAMERA_TYPE)
    cam = Webcam(image_w=cfg.IMAGE_W, image_h=cfg.IMAGE_H, image_d=cfg.IMAGE_DEPTH)
    V.add(cam, inputs=[], outputs=['cam/image_array'], threaded=True)

    # 添加Web控制器
    ctr = LocalWebController(port=cfg.WEB_CONTROL_PORT, mode='user')
    V.add(ctr,
          inputs=['cam/image_array'],
          outputs=['user/steering', 'user/throttle', 'user/mode', 'recording', 'web/buttons'],
          threaded=True)

    # 转换user/steering为user/angle
    V.add(Pipe(), inputs=['user/steering'], outputs=['user/angle'])

    # 直接将用户输入作为控制输出
    V.add(Pipe(), inputs=['user/angle'], outputs=['steering'])
    V.add(Pipe(), inputs=['user/throttle'], outputs=['throttle'])

    # 添加电调舵机控制
    if cfg.DRIVE_TRAIN_TYPE == "PWM_STEERING_THROTTLE":
        dt = cfg.PWM_STEERING_THROTTLE
        
        steering_controller = PulseController(
            pwm_pin=pins.pwm_pin_by_id(dt["PWM_STEERING_PIN"]),
            pwm_scale=dt["PWM_STEERING_SCALE"],
            pwm_inverted=dt["PWM_STEERING_INVERTED"])
        steering = PWMSteering(controller=steering_controller,
                                left_pulse=dt["STEERING_LEFT_PWM"],
                                right_pulse=dt["STEERING_RIGHT_PWM"])

        throttle_controller = PulseController(
            pwm_pin=pins.pwm_pin_by_id(dt["PWM_THROTTLE_PIN"]),
            pwm_scale=dt["PWM_THROTTLE_SCALE"],
            pwm_inverted=dt['PWM_THROTTLE_INVERTED'])
        throttle = PWMThrottle(controller=throttle_controller,
                               max_pulse=dt['THROTTLE_FORWARD_PWM'],
                               zero_pulse=dt['THROTTLE_STOPPED_PWM'],
                               min_pulse=dt['THROTTLE_REVERSE_PWM'])

        V.add(steering, inputs=['steering'], threaded=True)
        V.add(throttle, inputs=['throttle'], threaded=True)

    print("You can now go to <your hostname.local>:%d to drive your car." % cfg.WEB_CONTROL_PORT)
    
    V.start(rate_hz=cfg.DRIVE_LOOP_HZ, max_loop_count=cfg.MAX_LOOPS)

if __name__ == '__main__':
    cfg = dk.load_config()
    drive_simple(cfg)
